/* Copyright (c) 2009 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opensocial.services;

import org.opensocial.Request;
import org.opensocial.models.Activity;

import java.util.HashMap;
import java.util.Map;

/**
 * OpenSocial API class for activity requests; contains static methods for
 * fetching and creating activities.
 * 
 * @author Jason Cooper
 */
public class ActivitiesService extends Service {
//"/{userId}+/{groupId}/{appId}/{activityId}+")
  private static final String restTemplate = "activities/{guid}/{selector}/{appid}/{activityid}";

  /**
   * Returns a new Request instance which, when submitted, fetches the current
   * viewer's activities previously generated by the current application and
   * makes this data available as a List of Activity objects. Equivalent to
   * getActivities("@me").
   * 
   * @return new Request object to fetch the current viewer's activities
   * @see Activity
   */
  public static Request getActivities() {
    return getActivities(ME);
  }

  /**
   * Returns a new Request instance which, when submitted, fetches the specified
   * user's activities previously generated by the current application and makes
   * this data available as a List of Activity objects.
   * 
   * @param guid
   *          OpenSocial ID of the user whose activities are to be fetched
   * @return new Request object to fetch the specified user's activities
   * @see Activity
   */
  public static Request getActivities(String guid) {
    Request request = new Request(restTemplate, "activities.get", "GET");
    request.setModelClass(Activity.class);
    request.setSelector(SELF);
    request.setAppId(APP);
    request.setGuid(guid);

    return request;
  }

  /**
   * Returns a new Request instance which, when submitted, fetches the
   * activities previously generated by the current application for the
   * specified user's friends and makes this data available as a List of
   * Activity objects. Pass "@me" to fetch activities for all friends of the
   * current viewer.
   * 
   * @param guid
   *          OpenSocial ID of the user whose friends' activities are to be
   *          fetched
   * @return new Request object to fetch the specified user's friends'
   *         activities
   * @see Activity
   */
  public static Request getFriendActivities(String guid) {
    Request request = new Request(restTemplate, "activities.get", "GET");
    request.setModelClass(Activity.class);
    request.setSelector(FRIENDS);
    request.setAppId(APP);
    request.setGuid(guid);

    return request;
  }

  /**
   * Returns a new Request instance which, when submitted, generates a new
   * activity for the current application associated with the current viewer.
   * 
   * @param activity
   *          Activity object specifying the activity parameters to pass into
   *          the request; typically, title, titleId, and body are set
   * @return new Request object o create a new activity for the current viewer
   * @see Activity
   */
  public static Request createActivity(Activity activity) {
    Request request = new Request(restTemplate, "activities.create", "POST");
    request.setSelector(SELF);
    request.setAppId(APP);
    request.setGuid(ME);

    // Add RPC payload parameters
    Map activityParameter = new HashMap();
    String[] fieldNames = activity.getFieldNames();
    for (String fieldName : fieldNames) {
      activityParameter.put(fieldName, activity.getField(fieldName));
    }
    request.addRpcPayloadParameter("activity", activityParameter);

    // Add REST payload parameters
    request.addRestPayloadParameters(activity);

    return request;
  }
}
