/* Copyright (c) 2009 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opensocial.services;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.opensocial.Request;
import org.opensocial.models.Message;
import org.opensocial.models.MessageCollection;

/**
 * OpenSocial API class for message requests; contains static methods for
 * fetching and creating messages.
 * 
 * @author Okke Harsta
 */
public class MessagesService extends Service {

 // private static final String restTemplate = "messages/{guid}/{selector}/{appid}/{messageIds}";
  private static final String restTemplate = "messages/{guid}/{selector}";

  /**
   * Returns a new Request instance which, when submitted, fetches the current
   * viewer's activities previously generated by the current application and
   * makes this data available as a List of Message objects. Equivalent to
   * getMessages("@me").
   * 
   * @return new Request object to fetch the current viewer's activities
   * @see Message
   */
  public static Request getMessages() {
    return getMessages(ME);
  }

  /**
   * Returns a new Request instance which, when submitted, fetches the specified
   * user's activities previously generated by the current application and makes
   * this data available as a List of Message objects.
   * 
   * @param guid
   *          OpenSocial ID of the user whose messages are to be fetched
   * @return new Request object to fetch the specified user's messages
   * @see Message
   */
  public static Request getMessages(String guid) {
    Request request = new Request(restTemplate, "messages.get", "GET");
    request.setModelClass(Message.class);
    request.setSelector(SELF);
    request.setAppId(APP);
    request.setGuid(guid);

    return request;
  }

  
    /**
   * Returns a new Request instance which, when submitted, generates a new
   * message for the current application associated with the current viewer.
   * 
   * @param message
   *          Activity object specifying the activity parameters to pass into
   *          the request; typically, title, titleId, and body are set
   * @return new Request object o create a new message for the current viewer
   * @see Message
   */
  public static Request createMessage(Message message) {
    Request request = new Request(restTemplate, "messages.create", "POST");
    List collectionIds = message.getFieldAsList("collectionIds");
    if (collectionIds == null || collectionIds.isEmpty()) {
      throw new IllegalArgumentException("A messageCollectionId must be provided when posting a message");
    }
    request.setSelector((String) collectionIds.get(0));
    request.setAppId(APP);
    request.setGuid(ME);
    request.setModelClass(Message.class);

    // Add RPC payload parameters
    Map messageParameter = new HashMap();
    String[] fieldNames = message.getFieldNames();
    for (String fieldName : fieldNames) {
      messageParameter.put(fieldName, message.getField(fieldName));
    }
    request.addRpcPayloadParameter("message", messageParameter);

    // Add REST payload parameters
    request.addRestPayloadParameters(message);

    return request;
  }
  
  /**
   * Returns a new Request instance which, when submitted, generates a new
   * messageCollection for the current application associated with the current viewer.
   * 
   * @param messageCollection
   *          MessageCollection object specifying the activity parameters to pass into
   *          the request; only title is set
   * @return new Request object o create a new message for the current viewer
   * @see MessageCollection
   */
  public static Request createMessageCollection(MessageCollection messageCollection) {
    Request request = new Request(restTemplate, "messages.create", "POST");
    //request.setSelector(SELF);
    request.setSelector(null);
    request.setAppId(APP);
    request.setGuid(ME);
    request.setModelClass(MessageCollection.class);

    // Add RPC payload parameters
    Map messageParameter = new HashMap();
    String[] fieldNames = messageCollection.getFieldNames();
    for (String fieldName : fieldNames) {
      messageParameter.put(fieldName, messageCollection.getField(fieldName));
    }
    request.addRpcPayloadParameter("messageCollection", messageParameter);

    // Add REST payload parameters
    request.addRestPayloadParameters(messageCollection);

    return request;
  }
}
