/*
  $Id: LdifResultConverter.java 1330 2010-05-23 22:10:53Z dfisher $

  Copyright (C) 2003-2010 Virginia Tech.
  All rights reserved.

  SEE LICENSE FOR MORE INFORMATION

  Author:  Middleware Services
  Email:   middleware@vt.edu
  Version: $Revision: 1330 $
  Updated: $Date: 2010-05-23 18:10:53 -0400 (Sun, 23 May 2010) $
*/
package edu.vt.middleware.ldap.ldif;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import javax.naming.NamingException;
import edu.vt.middleware.ldap.bean.LdapBeanFactory;
import edu.vt.middleware.ldap.bean.LdapBeanProvider;
import edu.vt.middleware.ldap.bean.LdapResult;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * <code>LdifResultConverter</code> provides utility methods for converting
 * <code>LdapResult</code> to and from LDIF in string format.
 *
 * @author  Middleware Services
 * @version  $Revision: 1330 $ $Date: 2010-05-23 18:10:53 -0400 (Sun, 23 May 2010) $
 */
public class LdifResultConverter
{

  /** Log for this class. */
  protected final Log logger = LogFactory.getLog(getClass());

  /** Ldap bean factory. */
  protected LdapBeanFactory beanFactory = LdapBeanProvider.getLdapBeanFactory();

  /** Class for outputting LDIF. */
  private Ldif ldif = new Ldif();


  /**
   * Returns the factory for creating ldap beans.
   *
   * @return  <code>LdapBeanFactory</code>
   */
  public LdapBeanFactory getLdapBeanFactory()
  {
    return this.beanFactory;
  }


  /**
   * Sets the factory for creating ldap beans.
   *
   * @param  lbf  <code>LdapBeanFactory</code>
   */
  public void setLdapBeanFactory(final LdapBeanFactory lbf)
  {
    if (lbf != null) {
      this.beanFactory = lbf;
      this.ldif.setLdapBeanFactory(lbf);
    }
  }


  /**
   * This returns this <code>LdifResult</code> as LDIF.
   *
   * @param  result  <code>LdapResult</code> to convert
   *
   * @return  <code>String</code>
   */
  public String toLdif(final LdapResult result)
  {
    final StringWriter writer = new StringWriter();
    try {
      this.ldif.outputLdif(result.toSearchResults().iterator(), writer);
    } catch (IOException e) {
      if (this.logger.isWarnEnabled()) {
        this.logger.warn("Could not write ldif to StringWriter", e);
      }
    }
    return writer.toString();
  }


  /**
   * This reads any entries in the supplied LDIF into this <code>
   * LdifResult</code>.
   *
   * @param  ldif  <code>String</code> to read
   *
   * @return  <code>LdapResult</code>
   */
  public LdapResult fromLdif(final String ldif)
  {
    final LdapResult result = this.beanFactory.newLdapResult();
    try {
      result.addEntries(this.ldif.importLdif(new StringReader(ldif)));
    } catch (IOException e) {
      if (this.logger.isWarnEnabled()) {
        this.logger.warn("Could not read ldif from StringReader", e);
      }
    } catch (NamingException e) {
      if (this.logger.isErrorEnabled()) {
        this.logger.error("Unexpected naming exception occurred", e);
      }
    }
    return result;
  }
}
