/*
 *
 * ====================================================================
 *
 * The P6Spy Software License, Version 1.1
 *
 * This license is derived and fully compatible with the Apache Software
 * license, see http://www.apache.org/LICENSE.txt
 *
 * Copyright (c) 2001-2002 Andy Martin, Ph.D. and Jeff Goke
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in
 * the documentation and/or other materials provided with the
 * distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 * any, must include the following acknowlegement:
 * "The original concept and code base for P6Spy was conceived
 * and developed by Andy Martin, Ph.D. who generously contribued
 * the first complete release to the public under this license.
 * This product was due to the pioneering work of Andy
 * that began in December of 1995 developing applications that could
 * seamlessly be deployed with minimal effort but with dramatic results.
 * This code is maintained and extended by Jeff Goke and with the ideas
 * and contributions of other P6Spy contributors.
 * (http://www.p6spy.com)"
 * Alternately, this acknowlegement may appear in the software itself,
 * if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "P6Spy", "Jeff Goke", and "Andy Martin" must not be used
 * to endorse or promote products derived from this software without
 * prior written permission. For written permission, please contact
 * license@p6spy.com.
 *
 * 5. Products derived from this software may not be called "P6Spy"
 * nor may "P6Spy" appear in their names without prior written
 * permission of Jeff Goke and Andy Martin.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package com.p6spy.engine.logging.appender;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.p6spy.engine.common.FastExternalUtils;

// log4j, stdout, and file logger all use the same format
// so we descend from this class.
// also, they all have to do the get and setLastEntry stuff
// so we go ahead and do the work here.
public abstract class FormattedLogger {
    protected String lastEntry;

    /**
     * get a couple of lines of stack
     * @return the couple of lines of stack
     */
    public static String stack() {
      Exception exception = new Exception();
      //get the stack
      String stack = FastExternalUtils.getFullStackTrace(exception);
      //split into an array of strings (on newline)
      String[] stackLines = FastExternalUtils.splitTrim(stack, "\n");
      
      StringBuffer result = new StringBuffer();
      
      boolean foundOne = false;
      
      for (int i=0;i<stackLines.length;i++) {
        String current = stackLines[i];
        if (!restrictStack(current)) {
          if (!foundOne) {
            //put the FAST call in there
            if (i !=0) {
              result.append(extractFileLine(stackLines[i-1]));
            }
            foundOne = true;
          }
          if (result.length() > 0) {
            result.append(", ");
          }
          result.append(extractFileLine(current));
        }
      }
      return result.toString();
    }
    
    /**
     * see if we should keep moving down the stack until we see what called hibernate
     * @param line
     * @return true if should keep moving, false if there
     */
    private static boolean restrictStack(String line) {
      if (line == null || line.length() == 0) {
        return true;
      }
      if (line.startsWith("java.lang.Exception")) {
        return true;
      }
      if (line.startsWith("at com.p6spy")) {
        return true;
      }
      if (line.startsWith("at org.hibernate")) {
        return true;
      }
      if (line.startsWith("at edu.upenn.isc.fast.hibernate")) {
        return true;
      }
      if (line.startsWith("at edu.upenn.isc.fast.beans.codeGeneration.DataObjectBase")) {
        return true;
      }
      if (line.startsWith("at edu.internet2.middleware.grouper.hibernate")) {
          return true;
      }
      //make sure its an isc stack or internet2
      return !line.startsWith("at edu.upenn")  && !line.startsWith("at edu.internet2");
    }
    
    /**
     * pattern to get the file, method, and line number:
     * ^.*    beginning
     * \\.    dot
     * (.+) method
     * \\(    open paren
     * (.+)   file name
     * :      colon
     * (\\d+) line number
     * \\)    close paren
     * .*$      end
     */
    private static Pattern extractFileLinePattern = Pattern.compile("^.*\\.(.+)\\((.+):(\\d+)\\).*$"); 
    
    /**
     * get the good part out of the full line (or just return it if the parsing doesnt work)
     * @param fullLine
     * @return the file, method, line
     */
    private static String extractFileLine(String fullLine) {
      Matcher matchResult = extractFileLinePattern.matcher(fullLine);

      if (matchResult.find()) {
        String method = matchResult.group(1);
        String file = matchResult.group(2);
        String line = matchResult.group(3);
        return file + "." + method + "() line " + line;
      }
      return fullLine;
    }
    
    /**
     * 061128 CH: improve the format
     * @param connectionId
     * @param now
     * @param elapsed
     * @param category
     * @param prepared
     * @param sql
     */
    public void logSQL(int connectionId, String now, long elapsed, String category, String prepared, String sql) {
      //dont log select 1 from dual
      if ("SELECT 1 FROM DUAL".equals(sql)) {
        return;
      }
      boolean printStack = "statement".equals(category);
      
      String theConnectionId = ""; //connectionId==-1 ? "" : (" id: " + String.valueOf(connectionId) + ",");
      String theElapsed = elapsed == -1? "" : (", "+ elapsed + "ms");
      String theSql = (sql == null || sql.length() == 0) ? "" : (sql);
      String logEntry = now + theElapsed + theConnectionId +", "+category;
      String stack = null;
      if (printStack) {
        stack = stack();
      }
      if (stack != null && stack.length() > 0) {
        logEntry = now + theElapsed 
          + theConnectionId +", "+category+ ": " + stack;
        if (theSql.length() > 0) {
          logEntry += "\n   " + theSql;
        }
      } else {
        //put a colon after the category
        if (theSql.length() > 0) {
          theSql = ": " + theSql;
        }
        logEntry = now + theElapsed 
          + theConnectionId +", "+category+theSql;
      }
      logText(logEntry);
    }

    public abstract void logText(String text);

    // they also all need to have the last entry thing
    public void setLastEntry(String inVar) {
	lastEntry = inVar;
    }

    public String getLastEntry() {
	return lastEntry;
    }
}

