/*
 *
 * ====================================================================
 *
 * The P6Spy Software License, Version 1.1
 *
 * This license is derived and fully compatible with the Apache Software
 * license, see http://www.apache.org/LICENSE.txt
 *
 * Copyright (c) 2001-2002 Andy Martin, Ph.D. and Jeff Goke
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in
 * the documentation and/or other materials provided with the
 * distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 * any, must include the following acknowlegement:
 * "The original concept and code base for P6Spy was conceived
 * and developed by Andy Martin, Ph.D. who generously contribued
 * the first complete release to the public under this license.
 * This product was due to the pioneering work of Andy
 * that began in December of 1995 developing applications that could
 * seamlessly be deployed with minimal effort but with dramatic results.
 * This code is maintained and extended by Jeff Goke and with the ideas
 * and contributions of other P6Spy contributors.
 * (http://www.p6spy.com)"
 * Alternately, this acknowlegement may appear in the software itself,
 * if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "P6Spy", "Jeff Goke", and "Andy Martin" must not be used
 * to endorse or promote products derived from this software without
 * prior written permission. For written permission, please contact
 * license@p6spy.com.
 *
 * 5. Products derived from this software may not be called "P6Spy"
 * nor may "P6Spy" appear in their names without prior written
 * permission of Jeff Goke and Andy Martin.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/**
 * Description: JDBC Driver Extension implementing PreparedStatement.
 *
 * $Author: mchyzer $
 * $Revision: 1.2 $
 * $Date: 2006/11/28 18:06:24 $
 *
 * $Id: P6PreparedStatement.java,v 1.2 2006/11/28 18:06:24 mchyzer Exp $
 *
 *
 *
 */

package com.p6spy.engine.spy;

import java.io.*;
import java.sql.*;
import java.text.SimpleDateFormat;
import java.math.*;


public class P6PreparedStatement extends P6Statement implements PreparedStatement {
    
    
    public final static int P6_MAX_FIELDS = 32;
    public static int P6_GROW_MAX = 32;
    protected PreparedStatement prepStmtPassthru;
    protected String preparedQuery;
    protected Object values[];
    protected boolean isString[];
    
    public P6PreparedStatement(P6Factory factory, PreparedStatement statement, P6Connection conn, String query) {
        super(factory, statement, conn);
        prepStmtPassthru = statement;
        this.preparedQuery = query;
        initValues();
    }
    
    protected void initValues() {
        values = new Object[P6_MAX_FIELDS+1];
        isString = new boolean[P6_MAX_FIELDS+1];
    }
    
    public void addBatch() throws SQLException {
        prepStmtPassthru.addBatch();
    }
    
    public void clearParameters() throws SQLException {
        prepStmtPassthru.clearParameters();
    }
    
    public boolean execute() throws SQLException {
        return prepStmtPassthru.execute();
    }
    
    public ResultSet executeQuery() throws SQLException {
        ResultSet resultSet = prepStmtPassthru.executeQuery();
        return (getP6Factory().getResultSet(resultSet, this, preparedQuery, getQueryFromPreparedStatement()));
    }
    
    public int executeUpdate() throws SQLException {
        return prepStmtPassthru.executeUpdate();
    }
    
    public ResultSetMetaData getMetaData() throws SQLException {
        return prepStmtPassthru.getMetaData();
    }
    
    public void setArray(int p0, Array p1) throws SQLException {
        setObjectAsString(p0, p1);
        // we need to make sure we get the real object in this case
        if (p1 instanceof P6Array) {
            prepStmtPassthru.setArray(p0,((P6Array)p1).passthru);
        } else{
            prepStmtPassthru.setArray(p0,p1);
        }
    }
    
    public void setAsciiStream(int p0, InputStream p1, int p2) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setAsciiStream(p0,p1,p2);
    }
    
    public void setBigDecimal(int p0, BigDecimal p1) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setBigDecimal(p0,p1);
    }

    public void setBinaryStream(int p0, InputStream p1, int p2) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setBinaryStream(p0,p1,p2);
    }
    
    public void setBlob(int p0, Blob p1) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setBlob(p0,p1);
    }
    
    public void setBoolean(int p0, boolean p1) throws SQLException {
        setObjectAsString(p0, new Boolean(p1));
        prepStmtPassthru.setBoolean(p0,p1);
    }
    
    public void setByte(int p0, byte p1) throws SQLException {
        setObjectAsString(p0, new Byte(p1));
        prepStmtPassthru.setByte(p0,p1);
    }
    
    public void setBytes(int p0, byte[] p1) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setBytes(p0,p1);
    }
    
    public void setCharacterStream(int p0, Reader p1, int p2) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setCharacterStream(p0,p1,p2);
    }
    
    public void setClob(int p0, Clob p1) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setClob(p0,p1);
    }
    
    public void setDate(int p0, Date p1) throws SQLException {
        setObjectAsDate(p0, p1);
        prepStmtPassthru.setDate(p0,p1);
    } 
    
    public void setDate(int p0, Date p1, java.util.Calendar p2) throws SQLException {
        setObjectAsDate(p0, p1);
        prepStmtPassthru.setDate(p0,p1,p2);
    }
    
    public void setDouble(int p0, double p1) throws SQLException {
        setObjectAsInt(p0, new Double(p1));
        prepStmtPassthru.setDouble(p0,p1);
    }
    
    public void setFloat(int p0, float p1) throws SQLException {
        setObjectAsInt(p0, new Float(p1));
        prepStmtPassthru.setFloat(p0,p1);
    }
    
    public void setInt(int p0, int p1) throws SQLException {
        setObjectAsInt(p0, new Integer(p1));
        prepStmtPassthru.setInt(p0,p1);
    }
    
    public void setLong(int p0, long p1) throws SQLException {
        setObjectAsInt(p0, new Long(p1));
        prepStmtPassthru.setLong(p0,p1);
    }
    
    public void setNull(int p0, int p1, String p2) throws SQLException {
        setObjectAsString(p0, null);
        prepStmtPassthru.setNull(p0,p1,p2);
    }
    
    public void setNull(int p0, int p1) throws SQLException {
        setObjectAsString(p0, null);
        prepStmtPassthru.setNull(p0,p1);
    }
    
    public void setObject(int p0, Object p1, int p2, int p3) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setObject(p0,p1,p2,p3);
    }
    
    public void setObject(int p0, Object p1, int p2) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setObject(p0,p1,p2);
    }
    
    public void setObject(int p0, Object p1) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setObject(p0,p1);
    }
    
    public void setRef(int p0, Ref p1) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setRef(p0,p1);
    }
    
    public void setShort(int p0, short p1) throws SQLException {
        setObjectAsString(p0, new Short(p1));
        prepStmtPassthru.setShort(p0,p1);
    }
    
    public void setString(int p0, String p1) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setString(p0,p1);
    }
    
    public void setTime(int p0, Time p1, java.util.Calendar p2) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setTime(p0,p1,p2);
    }
    
    public void setTime(int p0, Time p1) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setTime(p0,p1);
    }
    
    /**
     * write a string to a string buffer and escape single quotes to double single quotes
     * also encapsulate the entire string in single quotes
     * @param input string input, can be null
     * @return the result
     */
    public static String escapeQuoteStringForOracle(String input) {
      if (input == null) {
        return null;
      }
      StringBuffer result = new StringBuffer();
      //current char we are on as we escape
      char curChar = 'a';
      //length of each string
      int stringLength = input.length();
      //escape the string into the string buffer, change ' to ''
      for (int i = 0; i < stringLength; i++) {
        curChar = input.charAt(i);
        if (curChar == '\'') {
          result.append('\'');
        }
        result.append(curChar);
      }
      return result.toString();
    }

    public void setTimestamp(int p0, Timestamp p1, java.util.Calendar p2) throws SQLException {
        setObjectAsTimestamp(p0, p1);
        prepStmtPassthru.setTimestamp(p0,p1,p2);
    }
    
    public void setTimestamp(int p0, Timestamp p1) throws SQLException {
        setObjectAsTimestamp(p0, p1);
        prepStmtPassthru.setTimestamp(p0,p1);
    }
    
    public void setUnicodeStream(int p0, InputStream p1, int p2) throws SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setUnicodeStream(p0,p1,p2);
    }
    
    /* we override this because the p6statement version will not be 
     * able to return the accurate prepared statement or query information
     */
    // bug 161: getResultSet() should return null if this is an update
    // count or there are not more result sets
    public java.sql.ResultSet getResultSet() throws java.sql.SQLException {
	ResultSet rs = passthru.getResultSet();
        return (rs == null) ? null : getP6Factory().getResultSet(rs, this, preparedQuery, getQueryFromPreparedStatement());
    }
    
    /*
     * P6Spy specific functionality
     */
    public final String getQueryFromPreparedStatement() {
        int len = preparedQuery.length();
        StringBuffer t = new StringBuffer(len * 2);
        
        if (values != null) {
            int i = 1, limit = 0, base = 0;
            
            while ((limit = preparedQuery.indexOf('?',limit)) != -1) {
                if (isString[i]) {
                    t.append(preparedQuery.substring(base,limit));
                    if (values[i] == null) {
                      t.append("null");
                    } else {
                      t.append("'");
                      t.append(values[i]);
                      t.append("'");
                    }
                } else {
                    t.append(preparedQuery.substring(base,limit));
                    t.append(values[i]);
                }
                i++;
                limit++;
                base = limit;
            }
            if (base < len) {
                t.append(preparedQuery.substring(base));
            }
        }
        
        return t.toString();
    }
    
    protected void growValues(int newMax) {
        int size = values.length;
        Object [] values_tmp = new Object[newMax + P6_GROW_MAX];
        boolean [] isString_tmp = new boolean[newMax + P6_GROW_MAX];
        System.arraycopy(values, 0, values_tmp,  0, size);
        values = values_tmp;
        System.arraycopy(isString, 0, isString_tmp, 0, size);
        isString = isString_tmp;
    }
    
    
    protected  void setObjectAsString(int i, Object o) {
        if (values != null) {
            if (i >= 0) {
	       if ( i >= values.length) {
		   growValues(i);
		}
                values[i] = escapeQuoteStringForOracle(o == null ? null : o.toString());
                isString[i]  = true;
            }
        }
    }
    
    /**
     * format including minutes and seconds: yyyy/MM/dd HH:mm:ss
     */
    public static final String DATE_FORMAT = "yyyy/MM/dd HH:mm:ss";

    /**
     * date format, make sure to synchronize
     */
    final static SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_FORMAT);

    /**
     * convert a date to a string
     * @param date
     * @return the string version of the date
     */
    public static synchronized String dateString(Date date) {
      if (date == null) {
        return "null";
      }
      return dateFormat.format(date);
    }
    
    /**
     * convert a timestamp to a string
     * @param timestamp 
     * @return the string version of the timestamp
     */
    public static synchronized String timestampString(Timestamp timestamp) {
      if (timestamp == null) {
        return "null";
      }
      String dateString = dateFormat.format(timestamp);
      //append the microseconds
      int micros = timestamp.getNanos() / 1000;
      String microsString = Integer.toString(micros);
      //pad with zeros until length 6
      while (microsString.length() < 6) {
        microsString = "0" + microsString;
      }
      String timestampString = dateString + "." + microsString;
      return timestampString;
    }

    protected void setObjectAsDate(int i, Date date) {
    if (values != null) {
      if (i >= 0) {
        if (i >= values.length) {
          growValues(i);
        }

        //yyyy/mm/dd HH24:MI:SS e.g. 2003/07/09 15:43:12
        if (date == null) {
          values[i] = "null";
        } else {
          values[i] = "to_date('" + dateString(date) + "', 'yyyy/mm/dd HH24:MI:SS')";
        }
        isString[i] = false;
      }
    }
  }

  protected void setObjectAsTimestamp(int i, Timestamp timestamp) {
    if (values != null) {
      if (i >= 0) {
        if (i >= values.length) {
          growValues(i);
        }
        //to_timestamp('2003/07/09 15:43:12.123456', 'yyyy/mm/dd HH24:MI:SS.ff')
        if (timestamp == null) {
          values[i] = "null";
        } else {
          values[i] = "to_timestamp('" + timestampString(timestamp) + "', 'yyyy/mm/dd HH24:MI:SS.ff6')";
        }
        isString[i] = false;
      }
    }
  }
  
    protected  void setObjectAsInt(int i, Object o) {
        if (values != null) {
            if (i >=0) {    
                if (i >= values.length) {
                    growValues(i);
                }
                values[i] = (o == null) ? null : o.toString();
                isString[i]  = false;
            } 
        }
    }

    // Since JDK 1.4
    public void setURL(int p0, java.net.URL p1) throws java.sql.SQLException {
        setObjectAsString(p0, p1);
        prepStmtPassthru.setURL(p0, p1);
    }
    
    // Since JDK 1.4
    public java.sql.ParameterMetaData getParameterMetaData() throws java.sql.SQLException {
        return(prepStmtPassthru.getParameterMetaData());
    }

    /**
     * Returns the underlying JDBC object (in this case, a
     * java.sql.PreparedStatement).
     * <p>
     * The returned object is a java.sql.Statement due
     * to inheritance reasons, so you'll need to cast 
     * appropriately.
     *
     * @return the wrapped JDBC object 
     */
    public Statement getJDBC() {
	Statement wrapped = (prepStmtPassthru instanceof P6Statement) ?
	    ((P6Statement) prepStmtPassthru).getJDBC() :
	    prepStmtPassthru;

	return wrapped;
    }

   public int getValuesLength() {
     return values.length;
    }
}
